// 08/01/14 Digimatic interface program
// RC0 is data in
// RC1 is clock in
// RC2 is request out (active high)
// RC3 is zero button (active high)
// RC4 is range select (high=bigger range)
// RC5 is meter PWM output

#include <built_in.h>

// Pin definitions
sbit SPC_DATA at PORTC.B0;
sbit SPC_CLOCK at PORTC.B1;
sbit SPC_REQ at PORTC.B2;
sbit SET_ZERO at PORTC.B3;
sbit SET_RANGE at PORTC.B4;
signed int current_spc_value;
signed int zero_value;

// String holders
char txt7[7];

// Displays a value on the meter, relative to
// center position. Valid range is from -100 to +100.
void update_meter(signed long int lrw_value)
{
 unsigned int pwm_value;
 lrw_value*=512;
 lrw_value/=100;
 if(lrw_value<-512) {lrw_value=-512;}        // clip if below lower limit
 else if(lrw_value>511) {lrw_value=511;}     // or above higher limit
 pwm_value=512+lrw_value;                    // calculate pwm value
 CCPR1L=(pwm_value >> 2) & 255;              // load MSBs
 DC1B1_bit=Lo(lrw_value).B1;                 // load bit 1
 DC1B0_bit=Lo(lrw_value).B0;                 // load bit 0
}

// Read value from gauge
// Returns a signed int, in units of 0.01mm (if mm is selected)
signed int read_spc()
{
 // Received data array
 // Note: index 0 isn't used, so that the digits
 // match 1-13 as in digimatic
 unsigned short int recdata[14]={0,0,0,0,0,0,0,0,0,0,0,0,0,0};
 unsigned short int digit; // used to count digits
 signed int result; // final result (four digits of reading)
 
 // Set request
 SPC_REQ=1;
 
 // Receive data and store in array
 for(digit=1;digit<14;digit++)
 {
  while (SPC_CLOCK==1);
  recdata[digit].B0=SPC_DATA;
  while (SPC_CLOCK==0);
  while (SPC_CLOCK==1);
  recdata[digit].B1=SPC_DATA;
  while (SPC_CLOCK==0);
  while (SPC_CLOCK==1);
  recdata[digit].B2=SPC_DATA;
  while (SPC_CLOCK==0);
  while (SPC_CLOCK==1);
  recdata[digit].B3=SPC_DATA;
  while (SPC_CLOCK==0);
 }
 
 // Clear request
 SPC_REQ=0;
 
 // Convert lower four digits (D8-D11) into a signed value
 result=(recdata[8]*1000)+(recdata[9]*100)+(recdata[10]*10)+recdata[11];
 if (recdata[5]==8) {result*=-1;}
 
 // Return value
 return result;
}

void main() {

     // Set up ports
     ANSEL=0;
     ANSELH=0;
     TRISC=0b00011011;
     PORTC=0;
     
     // Set up PWM output
     TRISC.B5=1;               // disable PWM output
     PR2=255;                  // set PWM period
     CCP1CON=0b00001100;       // configure CCP module
     CCPR1L=0;                 // set duty cycle
     TMR2IF_bit=0;             // clear IF
     T2CON=0b00000011;         // Set TMR2 prescale (x16)
     TMR2ON_bit=1;             // turn TMR2 on
     while (TMR2IF_bit==0) {}  // wait until overflows
     TRISC.B5=0;               // finally, enable PWM output

     // Initialse UART and wait
     UART1_Init(9600);
     Delay_ms(100);
     
     // Write welcome message
     UART1_Write_Text("Digimatic readout interface 08/01/14");
     UART1_Write(13);
     UART1_Write(13);
     
     // Set meter to zero
     update_meter(0);
     
     // Init zero value;
     zero_value=0;
     
     // Main loop
     // Tasks to do:
     // 1) Measure value from digimatic interface
     // 2) Check state of zero button - if it's pressed, then reset the zero
     // 3) Check high/low range switch and display result on meter
     // 4) Send result out on serial port
     while (1)
     {
           // Read meter
           current_spc_value=read_spc();
           
           // Update zero if required
           if(SET_ZERO) {zero_value=current_spc_value;}
           
           // Depending on state of range input, display value
           if(SET_RANGE) {update_meter(current_spc_value-zero_value);}
           else {update_meter((current_spc_value-zero_value)*10);}
           
           // Finally, output current value from gauge and the value less zero
           IntToStr(read_spc(),txt7);
           UART1_Write_Text(txt7);
           UART1_Write_Text(",");
           IntToStr(current_spc_value-zero_value,txt7);
           UART1_Write_Text(txt7);
           uart1_write(13);
           
           // Delay until next
           Delay_ms(50);
           
     }
}